/**
* \file: GstreamerCommon.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: CarPlay
*
* \author: Veeraiyan Chidambaram /RBEI/ECF3/ veeraiyan.chidambaram@in.bosch.com
*          J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <inttypes.h>

#include <adit_logging.h>
#include "GstreamerCommon.h"

#include <inttypes.h>
using namespace std;

LOG_IMPORT_CONTEXT(dipo_gst);

#define getErrAsStr(X) ((X == EINVAL) ? "EINVAL" : (X == EPERM) ? "EPERM" : "UNKNOWNERR")

namespace adit { namespace carplay
{

const char* GstreamerChannelNameStrings[] =
{
        "(null)",
        "video out",
        "main audio out",
        "alternate audio out",
        "main audio in"
};

/* PRQA: Lint Message 826: deactivation because casting mechanism of GObject
 * throws the finding */
/*lint -save -e826*/

static GMainLoop* messageBusLoop = nullptr;
static int messageBusRef = 0;
static pthread_mutex_t messageBusMutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_t messageBusThread = 0;

static void* mainLoop(void* inData);

GstreamerCommon::GstreamerCommon()
{
    bus = nullptr;
    pipeline = nullptr;
    partial = nullptr;
    channelName = GstreamerChannelName_Invalid;
    watchId = 0;

}

GstreamerCommon::~GstreamerCommon()
{
}

bool GstreamerCommon::CreatePipeline(const std::string& inName, const std::string& inLaunch,
        GstreamerChannelNames inChannelName)
{
    channelName = inChannelName;

    GError* error = nullptr;
    partial = gst_parse_bin_from_description(inLaunch.c_str(), TRUE, &error);
    if (error != nullptr)
    {
        LOG_ERROR((dipo_gst, "gst_parse_bin_from_description: %d %s", error->code, error->message));
        g_error_free(error);
        return false; /* ========== leaving function ========== */
    }

    LOGD_VERBOSE((dipo_gst, "pipeline created"));

    // create pipeline bin
    pipeline = gst_pipeline_new(inName.c_str());
    if (pipeline == nullptr)
    {
        LOG_ERROR((dipo_gst, "gst_pipeline_new failed"));
        return false; /* ========== leaving function ========== */
    }

    // add bus callback
    bus = gst_pipeline_get_bus(GST_PIPELINE(pipeline));
    if (bus != nullptr)
    {
        watchId = gst_bus_add_watch(bus, (GstBusFunc)busCallBack, (void*)this);
    }
    else
    {
        LOG_ERROR((dipo_gst, "gst_pipeline_get_bus failed"));
        return false; /* ========== leaving function ========== */
    }

    gst_object_ref(partial); // pipeline takes ownership of partial, retain reference
    if (!gst_bin_add(GST_BIN(pipeline), partial))
    {
        gst_object_unref(partial); // failed, so release partial
        LOG_ERROR((dipo_gst, "could not add configured part to the pipeline"));
        return false; /* ========== leaving function ========== */
    }

    return true;
}



void GstreamerCommon::StopPipeline()
{
    if (pipeline != nullptr)
    {
        auto ret = gst_element_set_state(GST_ELEMENT(pipeline), GST_STATE_NULL);
        if (ret == GST_STATE_CHANGE_FAILURE)
        {
            // silent warning
            LOGD_DEBUG((dipo_gst, "gst_element_set_state GST_STATE_NULL failed"));
        }

        gst_element_get_state(GST_ELEMENT(pipeline), NULL, NULL, 1);

        gst_object_unref(pipeline);
        pipeline = nullptr;
    }

    // remove bus callback
    if (watchId != 0)
    {
        g_source_remove(watchId);
        watchId = 0;
    }

    if (bus != nullptr)
    {
        gst_object_unref(bus);
        bus = nullptr;
    }

    if (partial != nullptr)
    {
        gst_object_unref(partial);
        partial = nullptr;
    }
}

int SetThreadPriority(const char* threadName, uint32_t priority)
{
    int err;
    struct sched_param schedParam;
    int schedPolicy;

    err = pthread_getschedparam(pthread_self(), &schedPolicy, &schedParam);
    if(err == 0)
    {
        schedParam.sched_priority = priority;
        err = pthread_setschedparam(pthread_self(), SCHED_FIFO, &schedParam);
        if(err != 0)
            LOG_ERROR((dipo_gst, "%s: set priority failed with error %s", threadName, getErrAsStr(err)));
    }
    else
    {
        LOG_ERROR((dipo_gst, "%s: get priority failed with error %s", threadName, getErrAsStr(err)));
    }

    return err;
}

gboolean busCallBack(GstBus* inBus, GstMessage* inMessage, gpointer inPointer)
{
    (void)inBus;

    auto me = static_cast<GstreamerCommon*>(inPointer);

    switch (GST_MESSAGE_TYPE(inMessage))
    {
    case GST_MESSAGE_ERROR:
        {
            gchar* debug;
            GError* err;
            gst_message_parse_error(inMessage, &err, &debug);
            LOG_ERROR((dipo_gst, "Gstreamer error in %s: %s",
                    GstreamerChannelNameStrings[me->channelName],
                    err->message));
            g_error_free(err);
            g_free(debug);
        }
        break;
    default:
        break;
    }

    return TRUE;
}

/*
 * \see RFC 3550 NTP
 */
uint64_t gstToNTP64(uint64_t inGstTs)
{
    uint64_t seconds = inGstTs / GST_SECOND;
    uint64_t nanoseconds = inGstTs - seconds;
    uint64_t fraction = nanoseconds * 0xffffffff / GST_SECOND;

    // NTP64 32 bit seconds, 32 bit fraction
    return (seconds << 32) | (fraction & 0xffffffff);

    // NTP32 16 bit seconds, 16 bit fraction
    // ((seconds & 0xffff) << 16) | ((fraction >> 16) & 0xffff);
}

GstCaps* createCapsFromAudioFormat(AudioFormatStruct inFormat)
{
    return gst_caps_new_simple("audio/x-raw-int",
            "endianness", G_TYPE_INT, 1234,
            "signed", G_TYPE_BOOLEAN, TRUE,
            "width", G_TYPE_INT, inFormat.BitsPerChannel,
            "depth", G_TYPE_INT, 16,
            "rate", G_TYPE_INT, inFormat.SampleRate,
            "channels", G_TYPE_INT, inFormat.Channels,
            NULL);
}

void MessageBusAddRef(int threadPrio)
{
    // one static shared main loop for the message bus
    pthread_mutex_lock(&messageBusMutex);

    messageBusRef ++;
    if (messageBusRef == 1)
    {
        if (0 != pthread_create(&messageBusThread, nullptr, mainLoop, (void*)&threadPrio))
        {
            messageBusThread = 0;
            LOG_ERROR((dipo_gst, "could not create Gstreamer message bus thread"));
        }
    }

    pthread_mutex_unlock(&messageBusMutex);
}

void MessageBusUnref()
{
    // one static shared main loop for the message bus
    pthread_mutex_lock(&messageBusMutex);

    messageBusRef --;
    if (messageBusRef == 0)
    {
        if (messageBusLoop != nullptr)
        {
            // TODO make sure the main loop is already started
            g_main_loop_quit(messageBusLoop);
            g_main_loop_unref(messageBusLoop);
            messageBusLoop = nullptr;
        }

        if (messageBusThread != 0)
        {
            pthread_join(messageBusThread, nullptr);
            messageBusThread = 0;
        }
    }

    pthread_mutex_unlock(&messageBusMutex);
}

void* mainLoop(void* inData)
{
    int *priority = (int*)inData;

    // set thread name
    prctl(PR_SET_NAME, "GstMainLoop", 0, 0, 0);

    SetThreadPriority("GstMainLoop", *priority);

    // create g_main_context
    GMainContext *context = g_main_context_new();
    messageBusLoop = g_main_loop_new(context, FALSE);
    if (messageBusLoop == nullptr) {
        LOG_ERROR((dipo_gst, "g_main_loop_new failed"));
        return nullptr;
    }

    g_main_context_push_thread_default(context);

    g_main_loop_run(messageBusLoop);

    // unref context
    g_main_context_pop_thread_default(context);
    g_main_context_unref(context);
    context = NULL;

    return nullptr;
}

uint64_t GetHostTicks()
{
    struct timespec ts;
    ts.tv_sec  = 0;
    ts.tv_nsec = 0;
    clock_gettime(CLOCK_MONOTONIC, &ts);
    return ((uint64_t)ts.tv_sec) * GST_SECOND + ts.tv_nsec;
}

uint64_t GetTickDifference(uint64_t inFirst, uint64_t inSecond)
{
    if (inFirst <= inSecond)
    {
        return inSecond - inFirst;
    }
    else
    {
        return inSecond + (((uint64_t)-1) - inFirst);
    }
}

} } // namespace adit { namespace carplay

/*lint -restore*/
